using System;
using System.Drawing;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;
using System.Text.RegularExpressions;
//
using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.GUI.controls;
using gov.va.med.vbecs.ExceptionManagement;
//
using System.Diagnostics;
//
namespace gov.va.med.vbecs.GUI
{

	#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		/// Retrieves  either a selected invoice or a new invoice entered by the user.
		///</summary>

		#endregion

	public class DlgShipmentOut : gov.va.med.vbecs.GUI.VbecsBaseForm
	{
		#region Variables

		private const string TT_ALL = "An invoice number, a ship to location, and a shipment type must be entered.";
		private const string TT_INVOICE_AND_SHIP_TO_LOCATION = "An invoice number and a ship to location must be entered.";
		private const string TT_INVOICE_AND_SHIPMENT_TYPE = "An invoice number and a shipment type must be entered.";
		private const string TT_SHIP_TO_LOCATION_AND_SHIPMENT_TYPE = "A ship to location and a shipment type must be entered.";
		private const string TT_INVOICE = "An invoice number must be entered.";
		private const string TT_SHIP_TO_LOCATION = "A ship to location must be entered.";
		private const string TT_SHIPMENT_TYPE = "A shipment type must be entered.";

		private bool _btnOKClicked;
		private bool _isNewInvoice = false;
       
		private Regex _regexInvoiceNo = new Regex(@"\w", RegexOptions.None);

        /// <summary>
        /// The invoice which we return to our caller.
        /// </summary>
        private BOL.OutgoingShipmentCollection _invoice;
		
		private DataTable _lvListSource;
		private DataTable _facilities;
        
        private string _invoiceNumber;
		
		private System.Guid _localSupplierGuid;
		private System.Guid _shipmentGuid;
		
		private DateTime _shipmentDate;
		
		private Common.ShipmentType _shipmentType;

		#endregion

		#region Form Controls

		private System.Windows.Forms.Button btnCancel;
		private System.Windows.Forms.GroupBox grpImage;
		private System.Windows.Forms.PictureBox pictureBox1;
		private System.Windows.Forms.Button btnNewInvoice;
		private System.Windows.Forms.Button btnExistingInvoice;
		private System.Windows.Forms.ListView lvInvoices;
		private System.Windows.Forms.GroupBox grpNewInvoice;
		private System.Windows.Forms.DateTimePicker dtpProcessingDT;
		private System.Windows.Forms.TextBox txtInvoice;
		private System.Windows.Forms.Label lblProcessingDT;
		private System.Windows.Forms.Label lblFacility;
		private System.Windows.Forms.Label lblInvoice;
		private System.Windows.Forms.GroupBox grpSelectInvoice;
		private System.Windows.Forms.ComboBox cmbFacility;
		private System.Windows.Forms.Label label1;
		private System.Windows.Forms.ColumnHeader colUpdateDate;
		private System.Windows.Forms.ColumnHeader colInvoice;
		private System.Windows.Forms.ColumnHeader colShipTo;
        private System.Windows.Forms.Button btnReview;
		private System.Windows.Forms.ColumnHeader columnHeader1;
		private gov.va.med.vbecs.GUI.controls.VbecsButton btnOK;
		private System.Windows.Forms.RadioButton rdoReturnToSupplier;
		private System.Windows.Forms.RadioButton rdoUnrelatedFacility;
		private System.ComponentModel.IContainer components;
    
		#endregion

		#region Constructors

		/// <summary>
		/// Private constructor for this dialog, initializes form variables;
		/// </summary>
		private DlgShipmentOut()
		{
            // Required for Windows Form Designer support
            //
            InitializeComponent();
            //
            SetComponentProperties();
        }

		/// <summary>
		/// Public constructor indicating whether we are after new or
		/// existing invoices.
		/// </summary>
		/// <param name="useNew"></param>
		public DlgShipmentOut(bool useNew) :this()
		{
			//
            this.IsNewInvoice = useNew; 
        }

		/// <summary>
		/// Public constructor passing owner Form and new indicator.
		/// </summary>
		/// <param name="mdiParent">Owner form</param>
		/// <param name="useNew">Indicator of new/existing invoice.</param>
		public DlgShipmentOut(Form mdiParent, bool useNew):this(useNew) 
		{
			//
			this.MdiParent = mdiParent;
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/10/2002</CreationDate>
		///
        /// <summary>
        /// Public constructor indicating new and passing DataTable of invoices for selection.
        /// </summary>
        /// <param name="useNew"></param>
        /// <param name="invoices"></param>
		public DlgShipmentOut(bool useNew, DataTable invoices) :this(useNew)
		{
			//
			this.InvoiceListSource = invoices;
		}

		#endregion

		/// <summary>
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#region Windows Form Designer generated code
		/// <summary>
		/// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.components = new System.ComponentModel.Container();
			this.AcceptButton = this.btnOK;
			this.CancelButton = this.btnCancel;
			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(DlgShipmentOut));
			this.btnCancel = new System.Windows.Forms.Button();
			this.grpImage = new System.Windows.Forms.GroupBox();
			this.pictureBox1 = new System.Windows.Forms.PictureBox();
			this.btnNewInvoice = new System.Windows.Forms.Button();
			this.btnExistingInvoice = new System.Windows.Forms.Button();
			this.colUpdateDate = new System.Windows.Forms.ColumnHeader();
			this.colInvoice = new System.Windows.Forms.ColumnHeader();
			this.colShipTo = new System.Windows.Forms.ColumnHeader();
			this.grpSelectInvoice = new System.Windows.Forms.GroupBox();
			this.btnReview = new System.Windows.Forms.Button();
			this.lvInvoices = new System.Windows.Forms.ListView();
			this.columnHeader1 = new System.Windows.Forms.ColumnHeader();
			this.grpNewInvoice = new System.Windows.Forms.GroupBox();
			this.rdoUnrelatedFacility = new System.Windows.Forms.RadioButton();
			this.rdoReturnToSupplier = new System.Windows.Forms.RadioButton();
			this.label1 = new System.Windows.Forms.Label();
			this.cmbFacility = new System.Windows.Forms.ComboBox();
			this.txtInvoice = new System.Windows.Forms.TextBox();
			this.lblFacility = new System.Windows.Forms.Label();
			this.lblInvoice = new System.Windows.Forms.Label();
			this.dtpProcessingDT = new System.Windows.Forms.DateTimePicker();
			this.lblProcessingDT = new System.Windows.Forms.Label();
			this.btnOK = new gov.va.med.vbecs.GUI.controls.VbecsButton();
			this.grpImage.SuspendLayout();
			this.grpSelectInvoice.SuspendLayout();
			this.grpNewInvoice.SuspendLayout();
			this.SuspendLayout();
			// 
			// vbecsLogo1
			// 
			this.vbecsLogo1.Name = "vbecsLogo1";
			this.vbecsLogo1.Size = new System.Drawing.Size(510, 30);
			this.vbecsLogo1.TabIndex = 4;
			// 
			// lblRequiredField
			// 
			this.lblRequiredField.Location = new System.Drawing.Point(0, 247);
			this.lblRequiredField.Name = "lblRequiredField";
			this.lblRequiredField.TabIndex = 6;
			// 
			// btnCancel
			// 
			this.btnCancel.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
			this.btnCancel.DialogResult = System.Windows.Forms.DialogResult.Cancel;
			this.btnCancel.Location = new System.Drawing.Point(428, 236);
			this.btnCancel.Name = "btnCancel";
			this.btnCancel.TabIndex = 3;
			this.btnCancel.Text = "&Cancel";
			this.ttToolTip.SetToolTip(this.btnCancel, "Cancel this function");  // **** 508 Compliance ***
			this.btnCancel.Click += new System.EventHandler(this.btnCancel_Click);
			// 
			// grpImage
			// 
			this.grpImage.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left)));
			this.grpImage.Controls.Add(this.pictureBox1);
			this.grpImage.Location = new System.Drawing.Point(4, 36);
			this.grpImage.Name = "grpImage";
			this.grpImage.Size = new System.Drawing.Size(56, 196);
			this.grpImage.TabIndex = 5;
			this.grpImage.TabStop = false;
			// 
			// pictureBox1
			// 
			this.pictureBox1.Image = ((System.Drawing.Image)(resources.GetObject("pictureBox1.Image")));
			this.pictureBox1.Location = new System.Drawing.Point(4, 12);
			this.pictureBox1.Name = "pictureBox1";
			this.pictureBox1.Size = new System.Drawing.Size(45, 45);
			this.pictureBox1.SizeMode = System.Windows.Forms.PictureBoxSizeMode.AutoSize;
			this.pictureBox1.TabIndex = 36;
			this.pictureBox1.TabStop = false;
			// 
			// ttToolTip
			// 
			this.ttToolTip.ShowAlways = true;
			// 
			// btnNewInvoice
			// 
			this.btnNewInvoice.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left)));
			this.btnNewInvoice.Location = new System.Drawing.Point(8, 168);
			this.btnNewInvoice.Name = "btnNewInvoice";
			this.btnNewInvoice.Size = new System.Drawing.Size(136, 23);
			this.btnNewInvoice.TabIndex = 1;
			this.btnNewInvoice.Text = "Create &New Invoice";
			this.btnNewInvoice.Visible = false;
			this.btnNewInvoice.Click += new System.EventHandler(this.btnNewInvoice_Click);
			// 
			// btnExistingInvoice
			// 
			this.btnExistingInvoice.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Left)));
			this.btnExistingInvoice.Location = new System.Drawing.Point(8, 168);
			this.btnExistingInvoice.Name = "btnExistingInvoice";
			this.btnExistingInvoice.Size = new System.Drawing.Size(136, 23);
			this.btnExistingInvoice.TabIndex = 9;
			this.btnExistingInvoice.Text = "Use &Existing Invoice";
			this.btnExistingInvoice.Click += new System.EventHandler(this.btnExistingInvoice_Click);
			// 
			// colUpdateDate
			// 
			this.colUpdateDate.Text = "Date Processed";
			this.colUpdateDate.Width = 130;
			// 
			// colInvoice
			// 
			this.colInvoice.Text = "Invoice";
			this.colInvoice.Width = 80;
			// 
			// colShipTo
			// 
			this.colShipTo.Text = "Ship To";
			this.colShipTo.Width = 100;
			// 
			// grpSelectInvoice
			// 
			this.grpSelectInvoice.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.grpSelectInvoice.Controls.Add(this.btnReview);
			this.grpSelectInvoice.Controls.Add(this.lvInvoices);
			this.grpSelectInvoice.Controls.Add(this.btnNewInvoice);
			this.grpSelectInvoice.Location = new System.Drawing.Point(64, 36);
			this.grpSelectInvoice.Name = "grpSelectInvoice";
			this.grpSelectInvoice.Size = new System.Drawing.Size(440, 196);
			this.grpSelectInvoice.TabIndex = 1;
			this.grpSelectInvoice.TabStop = false;
			this.grpSelectInvoice.Text = "Select Invoice*";
			// 
			// btnReview
			// 
			this.btnReview.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
			this.btnReview.Location = new System.Drawing.Point(272, 168);
			this.btnReview.Name = "btnReview";
			this.btnReview.Size = new System.Drawing.Size(160, 23);
			this.btnReview.TabIndex = 2;
			this.btnReview.Text = "&Review Confirmed Shipments";
			this.btnReview.Click += new System.EventHandler(this.btnReview_Click);
			// 
			// lvInvoices
			// 
			this.lvInvoices.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.lvInvoices.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
																						 this.colInvoice,
																						 this.colShipTo,
																						 this.colUpdateDate,
																						 this.columnHeader1});
			this.lvInvoices.FullRowSelect = true;
			this.lvInvoices.Location = new System.Drawing.Point(8, 16);
			this.lvInvoices.MultiSelect = false;
			this.lvInvoices.Name = "lvInvoices";
			this.lvInvoices.Size = new System.Drawing.Size(422, 148);
			this.lvInvoices.TabIndex = 0;
			this.lvInvoices.View = System.Windows.Forms.View.Details;
			this.lvInvoices.ColumnClick += new System.Windows.Forms.ColumnClickEventHandler(this.lvInvoices_ColumnClick);
			this.lvInvoices.SelectedIndexChanged += new System.EventHandler(this.lvInvoices_SelectedIndexChanged);
			// 
			// columnHeader1
			// 
			this.columnHeader1.Text = "Cancelled Date";
			this.columnHeader1.Width = 100;
			// 
			// grpNewInvoice
			// 
			this.grpNewInvoice.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.grpNewInvoice.BackColor = System.Drawing.SystemColors.Control;
			this.grpNewInvoice.Controls.Add(this.rdoUnrelatedFacility);
			this.grpNewInvoice.Controls.Add(this.rdoReturnToSupplier);
			this.grpNewInvoice.Controls.Add(this.label1);
			this.grpNewInvoice.Controls.Add(this.cmbFacility);
			this.grpNewInvoice.Controls.Add(this.txtInvoice);
			this.grpNewInvoice.Controls.Add(this.lblFacility);
			this.grpNewInvoice.Controls.Add(this.lblInvoice);
			this.grpNewInvoice.Controls.Add(this.btnExistingInvoice);
			this.grpNewInvoice.Controls.Add(this.dtpProcessingDT);
			this.grpNewInvoice.Controls.Add(this.lblProcessingDT);
			this.grpNewInvoice.Location = new System.Drawing.Point(64, 36);
			this.grpNewInvoice.Name = "grpNewInvoice";
			this.grpNewInvoice.Size = new System.Drawing.Size(440, 196);
			this.grpNewInvoice.TabIndex = 0;
			this.grpNewInvoice.TabStop = false;
			this.grpNewInvoice.Text = "New Shipment Invoice";
			// 
			// rdoUnrelatedFacility
			// 
			this.rdoUnrelatedFacility.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.rdoUnrelatedFacility.Location = new System.Drawing.Point(124, 96);
			this.rdoUnrelatedFacility.Name = "rdoUnrelatedFacility";
			this.rdoUnrelatedFacility.Size = new System.Drawing.Size(280, 20);
			this.rdoUnrelatedFacility.TabIndex = 6;
			this.rdoUnrelatedFacility.Text = "Send to &Unrelated Shipper";
			this.rdoUnrelatedFacility.CheckedChanged += new System.EventHandler(this.ShipmentTypeChanged);
			// 
			// rdoReturnToSupplier
			// 
			this.rdoReturnToSupplier.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.rdoReturnToSupplier.Location = new System.Drawing.Point(124, 76);
			this.rdoReturnToSupplier.Name = "rdoReturnToSupplier";
			this.rdoReturnToSupplier.Size = new System.Drawing.Size(284, 20);
			this.rdoReturnToSupplier.TabIndex = 5;
			this.rdoReturnToSupplier.TabStop = true;
			this.rdoReturnToSupplier.Text = "&Return to Shipper";
			this.rdoReturnToSupplier.CheckedChanged += new System.EventHandler(this.ShipmentTypeChanged);
			// 
			// label1
			// 
			this.label1.Location = new System.Drawing.Point(8, 76);
			this.label1.Name = "label1";
			this.label1.Size = new System.Drawing.Size(88, 20);
			this.label1.TabIndex = 4;
			this.label1.Text = "Shipment Type*";
			this.label1.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// cmbFacility
			// 
			this.cmbFacility.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.cmbFacility.DropDownStyle = System.Windows.Forms.ComboBoxStyle.DropDownList;
			this.cmbFacility.Location = new System.Drawing.Point(124, 48);
			this.cmbFacility.MaxDropDownItems = 4;
			this.cmbFacility.MaxLength = 50;
			this.cmbFacility.Name = "cmbFacility";
			this.cmbFacility.Size = new System.Drawing.Size(288, 22);
			this.cmbFacility.TabIndex = 3;
			this.cmbFacility.KeyUp += new System.Windows.Forms.KeyEventHandler(this.cmbFacility_KeyUp);
			this.cmbFacility.SelectedIndexChanged += new System.EventHandler(this.cmbFacility_SelectedIndexChanged);
			this.cmbFacility.AccessibleDescription = "Ship To";
			this.cmbFacility.AccessibleName = "Ship To";
			// 
			// txtInvoice
			// 
			this.txtInvoice.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.txtInvoice.Location = new System.Drawing.Point(124, 20);
			this.txtInvoice.MaxLength = 14;
			this.txtInvoice.Name = "txtInvoice";
			this.txtInvoice.Size = new System.Drawing.Size(288, 20);
			this.txtInvoice.TabIndex = 1;
			this.txtInvoice.Text = "";
			this.txtInvoice.Leave += new System.EventHandler(this.txtInvoice_Leave);
			this.txtInvoice.KeyUp += new System.Windows.Forms.KeyEventHandler(this.txtInvoice_KeyUp);
			// 
			// lblFacility
			// 
			this.lblFacility.Location = new System.Drawing.Point(8, 48);
			this.lblFacility.Name = "lblFacility";
			this.lblFacility.Size = new System.Drawing.Size(52, 20);
			this.lblFacility.TabIndex = 2;
			this.lblFacility.Text = "Ship To*";
			this.lblFacility.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// lblInvoice
			// 
			this.lblInvoice.Location = new System.Drawing.Point(8, 20);
			this.lblInvoice.Name = "lblInvoice";
			this.lblInvoice.Size = new System.Drawing.Size(92, 20);
			this.lblInvoice.TabIndex = 0;
			this.lblInvoice.Text = "Invoice Number*";
			this.lblInvoice.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// dtpProcessingDT
			// 
			this.dtpProcessingDT.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.dtpProcessingDT.CustomFormat = "";
			this.dtpProcessingDT.Format = System.Windows.Forms.DateTimePickerFormat.Custom;
			this.dtpProcessingDT.Location = new System.Drawing.Point(124, 120);
			this.dtpProcessingDT.Name = "dtpProcessingDT";
			this.dtpProcessingDT.Size = new System.Drawing.Size(288, 20);
			this.dtpProcessingDT.TabIndex = 8;
			this.dtpProcessingDT.Visible = false;
			this.dtpProcessingDT.ValueChanged += new System.EventHandler(this.dtpProcessingDT_ValueChanged);
			// 
			// lblProcessingDT
			// 
			this.lblProcessingDT.Location = new System.Drawing.Point(8, 120);
			this.lblProcessingDT.Name = "lblProcessingDT";
			this.lblProcessingDT.Size = new System.Drawing.Size(124, 20);
			this.lblProcessingDT.TabIndex = 7;
			this.lblProcessingDT.Text = "Processing Date*";
			this.lblProcessingDT.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			this.lblProcessingDT.Visible = false;
			// 
			// btnOK
			// 
			this.btnOK.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
			this.btnOK.DialogResult = System.Windows.Forms.DialogResult.OK;
			this.btnOK.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.btnOK.Image = ((System.Drawing.Image)(resources.GetObject("btnOK.Image")));
			this.btnOK.Location = new System.Drawing.Point(348, 236);
			this.btnOK.Name = "btnOK";
			this.btnOK.Size = new System.Drawing.Size(75, 23);
			this.btnOK.TabIndex = 2;
			this.btnOK.Click += new System.EventHandler(this.btnOK_Click);
			this.btnOK.MouseHover += new EventHandler(btnOK_MouseHover); // **** 508 Compliance ***
			// 
			// DlgShipmentOut
			// 
			this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
			this.ClientSize = new System.Drawing.Size(510, 263);
			this.KeyDownSpeakMessage += new KeyDownSpeakMessageEventHandler(DlgShipmentOut_KeyDownSpeakMessage); // **** 508 Compliance *** 
			this.Controls.Add(this.btnOK);
			this.Controls.Add(this.grpImage);
			this.Controls.Add(this.btnCancel);
			this.Controls.Add(this.grpNewInvoice);
			this.Controls.Add(this.grpSelectInvoice);
			this.hlphelpChm.SetHelpKeyword(this, "Outgoing_Shipment.htm");
			this.hlphelpChm.SetHelpNavigator(this, System.Windows.Forms.HelpNavigator.Topic);
			
			this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
			this.MaximizeBox = false;
			this.MinimizeBox = false;
			this.MinimumSize = new System.Drawing.Size(516, 288);
			this.Name = "DlgShipmentOut";
			this.RequiredFieldsLabel = true;
			this.hlphelpChm.SetShowHelp(this, true);
			this.StartPosition = System.Windows.Forms.FormStartPosition.CenterParent;
			this.Text = "VBECS - Process Outgoing Shipment";
			this.Load += new System.EventHandler(this.DlgShipmentOut_Load);
			this.Controls.SetChildIndex(this.grpSelectInvoice, 0);
			this.Controls.SetChildIndex(this.grpNewInvoice, 0);
			this.Controls.SetChildIndex(this.btnCancel, 0);
			this.Controls.SetChildIndex(this.grpImage, 0);
			this.Controls.SetChildIndex(this.vbecsLogo1, 0);
			this.Controls.SetChildIndex(this.lblRequiredField, 0);
			this.Controls.SetChildIndex(this.btnOK, 0);
			this.grpImage.ResumeLayout(false);
			this.grpSelectInvoice.ResumeLayout(false);
			this.grpNewInvoice.ResumeLayout(false);
			this.ResumeLayout(false);

		}
		#endregion

		#region properties

        /// <summary>
        /// Return/set whether we are returning a new or an existing invoice
        /// </summary>
		public bool IsNewInvoice
		{
			get 
			{ 
				return _isNewInvoice; 
			}
			set
			{
				_isNewInvoice = value;
				if (_isNewInvoice)
				{
					grpNewInvoice.Visible = true;
					grpSelectInvoice.Visible = false;
					btnNewInvoice.Visible = false;
					btnExistingInvoice.Visible = true;
					this.SetThisHelpFile(this, Common.StrRes.HelpFiles.UC011.DlgShipmentOut_New());
				}
				else
				{
					grpSelectInvoice.Visible = true;
					grpNewInvoice.Visible = false;
					btnNewInvoice.Visible = true;
					btnExistingInvoice.Visible = false;
					this.SetThisHelpFile(this, Common.StrRes.HelpFiles.UC011.DlgShipmentOut_Recent());
				}
			}
		}

		/// <summary>
		/// Constructor
		/// </summary>
		public bool BtnOKClicked
		{
			get
			{
				return this._btnOKClicked;
			}
		}

		/// <summary>
		/// Get/Set the list of existing invoices
		/// </summary>
		public DataTable InvoiceListSource
		{
			get
			{
				return _lvListSource;
			}
			set
			{
				_lvListSource = value;
				LoadInvoiceList();
			}
		}

		#endregion
	
		private void SetComponentProperties()
		{
			vbecsLogo1.Title=this.Text.Replace("VBECS - ","");
			// CR 2693
			// hlpHelp
			// 
			this.hlpHelp.HelpNamespace = Environment.CurrentDirectory+"\\VBECS.chm";
			// 
			// hlphelpChm
			// 
			this.hlphelpChm.HelpNamespace = Environment.CurrentDirectory+"\\VBECS.chm";
			// 
			//
			this.SetThisHelpFile(this, Common.StrRes.HelpFiles.UC011.DlgShipmentOut_New());
			//
			//set custom date and time format
			this.dtpProcessingDT.CustomFormat = Common.VBECSDateTime.DateTimeFormat;
			//set current date value based on the division time zone
			this.dtpProcessingDT.Value = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// User has decided to create a new invoice rather than use an existing one.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnNewInvoice_Click(object sender, System.EventArgs e)
		{
			IsNewInvoice = true;
			Init();
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// User has decided to process an existing invoice
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnExistingInvoice_Click(object sender, System.EventArgs e)
		{
			IsNewInvoice = false;
			Init();
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// Initialize control values;
		/// </summary>
		private void Init()
		{
			_invoiceNumber = "";
			_shipmentDate = DateTime.MinValue;
			_localSupplierGuid = Guid.Empty;
			_shipmentType = 0;
			_invoice = null;
			cmbFacility.SelectedIndex = -1;
			this.rdoReturnToSupplier.Checked = false;
			this.rdoUnrelatedFacility.Checked = false;
			txtInvoice.Text = "";
			dtpProcessingDT.Value = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			btnOK.EnabledButton = false;
			if ( IsNewInvoice )
			{
                txtInvoice.Focus();
			}
			else
			{

                lvInvoices.Focus();
                lvInvoices.Select();
			}
		}

		private void PrintShipment()
		{
			try
			{
				BOL.OutgoingShipmentCollection invoice = BOL.OutgoingShipmentCollection.GetOutgoingInvoiceByGuid(_invoice.ShipmentGuid);
				//
				DlgViewInvoice f = new DlgViewInvoice(invoice);
				//
				f.ShowDialogFromLockedParent(this, false);           //this.Parent??  TODO: LockingTest 
				//
			} 
			catch ( InvoiceException )
			{
				GuiMessenger.ShowMessageBox(this, Common.StrRes.SysErrMsg.UC011.UpdateError()); 
				this.Close();
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// User indicated acceptance of selected/entered values, return to caller.
		/// Implements BR_1.56
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnOK_Click(object sender, System.EventArgs e)
		{
			if (_isNewInvoice) 
			{ 
				this._invoiceNumber = txtInvoice.Text;
				this._shipmentDate = dtpProcessingDT.Value;
				//
				// BR_1.56: we need to make sure the invoice really is new
				if( BOL.OutgoingShipmentCollection.GetShipmentInvoiceCount(this._localSupplierGuid, this._invoiceNumber) > 0)
				{
					GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.Dynamic("The invoice number you have entered already exists for the chosen 'Ship To' location.  Please select a new one."));
					return;
				}
				else
				{ 
					this._invoice = BOL.OutgoingShipmentCollection.CreateOutgoingInvoice(this._invoiceNumber,
						this._localSupplierGuid,
						this._shipmentDate,
						this._shipmentType,
						ThisUser.UserNTLoginID, Common.UpdateFunction.UC011DlgShipmentOut);
				}
			}
			else
			{
				this._invoice = this.OutgoingShipment;
			}

			if( IsCurrentUserAuthorizedForFunction( Common.FunctionCode.StandardAccess ) )
			{
				if ( this._invoice.Confirmed && !this._isNewInvoice )
				{
					using ( new WaitCursor() ) 
					{
						PrintShipment();
						return;
					}
				}
				FrmProcessShipmentOut mainForm = new FrmProcessShipmentOut(this._isNewInvoice, this._invoice);

				// An error in the constructor will result in the form being Disposed.  We don't want to proceed if
				// this is the case
				if ( !mainForm.IsDisposed && mainForm.DialogResult != DialogResult.Abort )
				{
					mainForm.MdiParent = this.MdiParent;
					mainForm.Show();
					this.Close();
				}
			}
			else
			{
				GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.InsufficientSecurityPrivileges("perform this function"));
			}
		}


		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
        /// <summary>
        /// User has selected an existing invoice.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
		private void lvInvoices_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			if (lvInvoices.SelectedItems.Count > 0)
			{
				DataRow dr = (DataRow)lvInvoices.SelectedItems[0].Tag;
                try
                {
                    _invoice = BOL.OutgoingShipmentCollection.GetInvoiceFromDataRow(dr);
                }
                catch (BOL.InvoiceException ex)
                {
                    ExceptionManagement.ExceptionManager.Publish(ex);
					GuiMessenger.ShowMessageBox(this, Common.StrRes.SysErrMsg.UC011.InvalidData());
                    return;
                }
				_shipmentGuid = _invoice.ShipmentGuid;
				_invoiceNumber = _invoice.InvoiceNumber;
				_shipmentDate = _invoice.ShipmentDate;
                _shipmentType = _invoice.ShipmentType;
				_localSupplierGuid = _invoice.LocalSupplierGuid;
				ValidateSelection();
			}
			else
			{
				Init();
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// User is sorting the displayed list.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void lvInvoices_ColumnClick(object sender, System.Windows.Forms.ColumnClickEventArgs e)
		{
			if (e.Column == 2)
			{
				ColumnClickSorter.ColumnClickHandler(sender, e, ColumnClickSorter.ColumnSortType.DateTime);
			}
			else 
			{
				ColumnClickSorter.ColumnClickHandler(sender, e, ColumnClickSorter.ColumnSortType.String);
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// User is keying text in the facilities combobox to locate
		/// the ship destination.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void cmbFacility_KeyUp(object sender, System.Windows.Forms.KeyEventArgs e)
		{
			int index;
			//
			if ((e.KeyCode == Keys.Back) ||
				(e.KeyCode == Keys.Left) ||
				(e.KeyCode == Keys.Right) ||
				(e.KeyCode == Keys.Up) ||
				(e.KeyCode == Keys.Down) ||
				(e.KeyCode == Keys.Delete) ||
				(e.KeyCode == Keys.PageUp) ||
				(e.KeyCode == Keys.PageDown) ||
				(e.KeyCode == Keys.Home) ||
				(e.KeyCode == Keys.Control) ||
				(e.KeyCode == Keys.End))
			{
				return;
			}
			//
			index = this.cmbFacility.FindString(this.cmbFacility.Text);
			if (index > -1)
			{
				this.cmbFacility.SelectedIndex = index;
			}
		}

		/// <summary>
		/// Returns a generated invoice number based on the current date and time.
		/// </summary>
		/// <returns></returns>
		private string GenerateInvoiceNumber()
		{
			return BOL.VBECSDateTime.GetDivisionCurrentDateTime().ToString("yyyyMMddHmmss");
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// User has changed the contents of the invoice number field, Validate it.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void txtInvoice_TextChanged(object sender, System.EventArgs e)
		{
			_invoiceNumber = txtInvoice.Text.Trim();
            ValidateSelection();
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// Fill the facilities combobox for selection of shipment destination.
		/// </summary>
		private void FillFacilities()
		{
            cmbFacility.Items.Clear();
            int maxLen = 1;
            //
			// BR_11.10
			_facilities = BOL.LocalSupplier.GetLocalSuppliersForDivision();
			//
			foreach(DataRow dr in _facilities.Rows) 
			{
                maxLen = ((string)dr[Common.VbecsTables.Facility.FacilityName]).Length < maxLen ? maxLen : ((string)dr[Common.VbecsTables.Facility.FacilityName]).Length;
				cmbFacility.Items.Add(new ShipmentFacilityItem(dr)); 
			}
            //
            cmbFacility.MaxLength = maxLen;
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// Initialize required controls
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void DlgShipmentOut_Load(object sender, System.EventArgs e)
		{
			FillFacilities();
            Init();
		}
		/// <summary>
		/// User has changed the facility selection. If it's a new one,
		/// go get it and repopulate.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void cmbFacility_SelectedIndexChanged(object sender, System.EventArgs e)
		{
			_localSupplierGuid = Guid.Empty;

			if (cmbFacility.SelectedItem != null) 
			{
				Debug.WriteLine(Common.FunctionCode.AddNewLocalShipper.ToString());
				Debug.WriteLine(cmbFacility.SelectedItem.GetType().ToString());
				if ( (cmbFacility.SelectedItem.ToString() == "<New Receiving Facility>") && 
					(cmbFacility.SelectedIndex == (cmbFacility.Items.Count - 1)) &&
					(cmbFacility.SelectedItem.GetType().ToString() == "System.String"))
				{
					if ( ThisUser.IsAuthorizedForFunction( Common.FunctionCode.AddNewLocalShipper ) )
					{
						FrmCollectionFacilityMaintain frmNewColl = new FrmCollectionFacilityMaintain();
						frmNewColl.ShowDialogFromLockedParent(this, true);
						
						// After the user has entered a new facility, repopulate the combo box.  We cannot check
						// the button state from that form since the user always has to hit cancel to get out
						// of UC06 (even if they enter a new facility).
						FillFacilities();
						this.ValidateSelection();
						return;
						
					}
					else
					{
						GuiMessenger.ShowMessageBox(this, Common.StrRes.SysErrMsg.UC011.UserNotAuthorized());
						return;
					}
				}
				if (cmbFacility.SelectedItem.ToString() != "") 
				{
					_localSupplierGuid = (System.Guid)((ShipmentFacilityItem)cmbFacility.SelectedItem).RowValue[Common.VbecsTables.LocalSupplier.LocalSupplierGuid]; 
				}
				ValidateSelection();
			}
		}
		/// <summary>
		/// Unused
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void dtpProcessingDT_ValueChanged(object sender, System.EventArgs e)
		{
			_shipmentDate = dtpProcessingDT.Value;
		}
        /// <summary>
        /// If the user left the invoice number blank, he/she wants one generated.
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
		private void txtInvoice_Leave(object sender, System.EventArgs e)
		{
			// BR_11.02
				if ( txtInvoice.Text.Trim() == "" )
			{
				txtInvoice.Text = GenerateInvoiceNumber();
                _invoiceNumber = txtInvoice.Text.Trim();
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// Return the invoice the user has indicated.
		/// </summary>
		public BOL.OutgoingShipmentCollection OutgoingShipment
		{
			get
			{
				return _invoice;
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// Return the invoice number the user has entered.
		/// </summary>
		public string InvoiceNumber 
		{
			get 
			{
				return _invoiceNumber;
			}
		}
        /// <summary>
        /// Return the shipment type indicated/retrieved.
        /// </summary>
        public Common.ShipmentType ShipmentType
        {
            get 
            {
                return _shipmentType;
            }
            set 
            {
                _shipmentType = value;
            }
        }
		/// <summary>
		/// return the ship-to facility's guid.
		/// </summary>
		public System.Guid LocalSupplierGuid
		{
			get 
			{
				return _localSupplierGuid;
			}
		}
		/// <summary>
		/// return the Shipment row's Guid
		/// </summary>
		public System.Guid ShipmentGuid
		{
			get 
			{
				return _shipmentGuid;
			}
		}
		/// <summary>
		/// Unused.
		/// </summary>
		public DateTime ShipmentDate 
		{
			get 
			{
				return _shipmentDate;
			}
			set 
			{
				_shipmentDate = value;
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// Set/Get list of facilities for the combo.
		/// </summary>
		public DataTable FacilitySource
		{
			get 
			{
				return _facilities;
			}
			set 
			{
				_facilities = value;
				this.FillFacilities();
			}
		}

		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
		/// <summary>
		/// Get the list of existing shipments for this division
		/// and populate the list.
		/// </summary>
		private void LoadInvoiceList()
		{
			lvInvoices.Items.Clear();
			if (_lvListSource != null) 
			{
				try
				{
					//
					foreach(DataRow invoiceItem in _lvListSource.Rows)
					{
						ListViewItem lviInvoice = new ListViewItem();
						//
						if (_reviewConfirmed)
						{
							if (!invoiceItem.IsNull(Common.VbecsTables.Shipment.ShipmentDate))
							{
								lviInvoice.Text = invoiceItem[Common.VbecsTables.Shipment.InvoiceNumber].ToString();
								string facilityName = invoiceItem[Common.VbecsTables.Facility.FacilityName].ToString(); 
								lviInvoice.SubItems.Add(facilityName);
								lviInvoice.SubItems.Add(Common.VBECSDateTime.FormatDateTimeString(invoiceItem[Common.VbecsTables.Shipment.ShipmentDate]));
								lviInvoice.Tag = invoiceItem;
								lvInvoices.Items.Add(lviInvoice);
							}
						}
						else
						{
							if (invoiceItem.IsNull(Common.VbecsTables.Shipment.ShipmentDate))
							{
								lviInvoice.Text = invoiceItem[Common.VbecsTables.Shipment.InvoiceNumber].ToString();
								string facilityName = invoiceItem[Common.VbecsTables.Facility.FacilityName].ToString(); 
								lviInvoice.SubItems.Add(facilityName);
								lviInvoice.SubItems.Add(Common.VBECSDateTime.FormatDateTimeString(invoiceItem[Common.VbecsTables.Shipment.LastUpdateDate]));
								lviInvoice.Tag = invoiceItem;
								lvInvoices.Items.Add(lviInvoice);
							}
						}
					}
					VbecsFunction.SetColumnWidths(lvInvoices);
				}
				catch(Exception ex)
				{
					ExceptionManagement.ExceptionManager.Publish(ex);
					throw ex;
				}
			}
		}
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
        /// <summary>
        /// Validate presence of valid entries, enable/disable OK.
        /// </summary>
        private void ValidateSelection() 
        {
           if ((_localSupplierGuid != Guid.Empty) && 
                _regexInvoiceNo.IsMatch(_invoiceNumber) && 
                _shipmentType != Common.ShipmentType.Unknown)
            {
                btnOK.EnabledButton = true;
            }
            else
            {
                btnOK.EnabledButton = false;
            }
        }

        /// <summary>
        /// user is typing in the invoice textbox
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void txtInvoice_KeyUp(object sender, System.Windows.Forms.KeyEventArgs e)
        {
            _invoiceNumber = txtInvoice.Text.Trim();
            ValidateSelection();
        }

        bool _reviewConfirmed = false;

        private void btnReview_Click(object sender, System.EventArgs e)
        {
            _reviewConfirmed = !_reviewConfirmed;
            //
            if ( _reviewConfirmed )
            {
                btnReview.Text = "&Review Open Shipments";
                colUpdateDate.Text = "Date Shipped";
            }
            else
            {
                btnReview.Text = "&Review Confirmed Shipments";
                colUpdateDate.Text = "Date Last Modified";
            }
            LoadInvoiceList();
        }

		private void btnCancel_Click(object sender, System.EventArgs e)
		{
			this._btnOKClicked = false;
			this.Close();
		}

		private void ShipmentTypeChanged(object sender, System.EventArgs e)
		{
			if (this.rdoReturnToSupplier.Checked)
			{
				this.ShipmentType = OutgoingShipmentCollection.GetShipmentTypeFromChar('1');
			}
			else if (this.rdoUnrelatedFacility.Checked)
			{
				this.ShipmentType = OutgoingShipmentCollection.GetShipmentTypeFromChar('3');
			}
			else
			{
				this.ShipmentType = OutgoingShipmentCollection.GetShipmentTypeFromChar(char.MinValue);
			}
			//
			ValidateSelection();
		}
		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/21/2003</CreationDate>
		///
        /// <summary>
        /// Internal Class for the combobox.
        /// </summary>
        private class ShipmentFacilityItem
        {
            private DataRow _facilityRow;
            public ShipmentFacilityItem(DataRow dr)
            {
                _facilityRow = dr;
            }
            //
            public override string ToString()
            {
                return _facilityRow[Common.VbecsTables.Facility.FacilityName].ToString();
            }
            //
            public DataRow RowValue
            {
                get 
                {
                    return _facilityRow;
                }
                set 
                {
                    _facilityRow = value;
                }
            }
		}

		/// <summary>
		/// **** 508 Compliance ***
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void DlgShipmentOut_KeyDownSpeakMessage(object sender, EventArgs e)
		{
			string textToSpeak = GetToolTipForOKButton();
			//
			if ( textToSpeak != string.Empty )
			{
				this.SpeakMessage( textToSpeak );
			}
		}

		/// <summary>
		/// **** 508 Compliance ***
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnOK_MouseHover(object sender, EventArgs e)
		{
			string toolTip = GetToolTipForOKButton();
			//
			if ( toolTip != string.Empty )
			{
				this.ttToolTip.SetToolTip(btnOK, toolTip);
			}
		}

		/// <summary>
		/// **** 508 Compliance ***
		/// </summary>
		/// <returns></returns>
		private string GetToolTipForOKButton()
		{
			string toolTip = string.Empty;
			//
			if ( !this.btnOK.EnabledButton )
			{
				bool ttInvoiceRequired = this.txtInvoice.Text == null || this.txtInvoice.Text == string.Empty;
				bool ttShipToLocationRequired = this.cmbFacility.Text == null || this.cmbFacility.Text == string.Empty;
				bool ttShipmentTypeRequired = !this.rdoReturnToSupplier.Checked && !this.rdoUnrelatedFacility.Checked;
				//
				if ( ttInvoiceRequired && ttShipToLocationRequired && ttShipmentTypeRequired )
				{
					toolTip = TT_ALL;
				}
				else if ( ttInvoiceRequired && ttShipToLocationRequired )
				{
					toolTip = TT_INVOICE_AND_SHIP_TO_LOCATION;
				}
				else if ( ttInvoiceRequired && ttShipmentTypeRequired )
				{
					toolTip = TT_INVOICE_AND_SHIPMENT_TYPE;
				}
				else if ( ttShipToLocationRequired && ttShipmentTypeRequired )
				{
					toolTip = TT_SHIP_TO_LOCATION_AND_SHIPMENT_TYPE;
				}
				else if ( ttInvoiceRequired )
				{
					toolTip = TT_INVOICE;
				}
				else if ( ttShipToLocationRequired )
				{
					toolTip = TT_SHIP_TO_LOCATION;
				}
				else if ( ttShipmentTypeRequired )
				{
					toolTip = TT_SHIPMENT_TYPE;
				}
			}
			//
			return toolTip;
		}

	}
	
}
